{
THintListBox: an enhanced version of ListBox that shows "long items" (i.e. items
              that do not fit into its width) as an hint. Enjoy this code.


Author: Giuseppe Madaffari
Copyright:  2000 by Giuseppe Madaffari. All rights reserved.
Created: 21/01/2001
Modified: 12/02/2001
Version: 1.0
License: you can freely use and distribute the included code
         for any purpouse, but you cannot remove this copyright
         notice. Send me any comment and update, they are really
         appreciated. :-)
Contact: gmadaffari@infomedia.it
         www.programmers.net/artic/Madaffari/
         web.tiscalinet.it/gmadaffari
}

unit QHintListBox;

interface

uses
  SysUtils, Types, Classes, QGraphics, QControls, QForms, QDialogs,
  QStdCtrls, QExtCtrls;

type
  THintListBox = class(TListBox)
  private
    FHintLbl: TLabel;
    FTimer: TTimer;
    FIdx: integer;
    FShowHint: boolean;
    FHintHidePause: integer;
    procedure SetHintHidePause(const Value: integer);
    procedure SetShowPartialHint(const Value: boolean);
  protected
    // Internal handlers
    procedure DoOnClick(Sender: TObject);
    procedure DoOnTimer(Sender: TObject);
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
  published
    { Published declarations }
    property ShowPartialHint: boolean read FShowHint write SetShowPartialHint
      default True;
    property HintHidePause: integer read FHintHidePause write SetHintHidePause;
  end;

procedure Register;

implementation
uses QT;  // we need some low level call

const
  ScrollBarWidth = 20;  // should be default width for scrollbars

procedure Register;
begin
  RegisterComponents('Samples', [THintListBox]);
end;

{ THintListBox }

constructor THintListBox.Create(AOwner: TComponent);
begin
  inherited;
  FIdx := -1;  // -1 = no item 
  ShowPartialHint := True;
  HintHidePause := 1000;
end;

procedure THintListBox.DoOnClick(Sender: TObject);
begin
  // user clicked over hint label, therefore we have to select the item below...
  ItemIndex := FIdx;
  // ...and now we can hide the hint
  FHintLbl.Hide;
  FTimer.Enabled := False;
end;

procedure THintListBox.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  I: Integer;
  MinWidth: Integer;
  R: Trect;
begin
  inherited;
  // do we have to show an hint?
  if not FShowHint or (X > (ClientWidth - ScrollBarWidth)) then Exit;
  // let's figure out the item we are hoovering
  I := ItemAtPos(Point(X, Y), True);
  if I = -1 then Exit;
  if not ItemVisible(I) then
  begin
    FIdx := -1;
  end
  else if I <> FIdx then
  begin
    FHintLbl.Hide;
    FTimer.Enabled := False;
    FIdx := I;
    // compute item's rect
    R := ItemRect(FIdx);
    R.Right := Canvas.TextWidth (Items[FIdx]) + R.Left;
    MinWidth := ClientWidth;
    // is scrollbar visible?
    if QListBox_scrollBar(Handle) then
      Dec(MinWidth, ScrollBarWidth);

    // if item is smaller that listbox's width than do nothing
    if R.Right <= MinWidth then Exit;

    InflateRect(R, -1, -1);
    // setup the hint label
    with FHintLbl do
    begin
      Font := Self.Font;
      Color := Self.Color;
      // QT requires this call to follow the above in order to properly setup
      BorderStyle := bsSingle;
      Left := Self.Left + R.Left;
      Top := Self.Top + R.Top;
      Width := Canvas.TextWidth (Items[FIdx]) + 6;
      Height := ItemHeight;
      Caption := Items[FIdx];
      Parent := self.Parent;
      // show hint and activate the pause timer
      Show;
      FTimer.Enabled := True;
    end;
  end;
end;

procedure THintListBox.SetHintHidePause(const Value: integer);
begin
  FHintHidePause := Value;
  if Assigned(FTimer) then
    FTimer.Interval := FHintHidePause;
end;

procedure THintListBox.SetShowPartialHint(const Value: boolean);
begin
  if not (csDesigning in ComponentState) then
    if FShowHint and not Value then
    begin
      // we don't want hints, therefore let's save memory 
      FreeAndNil(FHintLbl);
      FreeAndNil(FTimer);
    end
    else if not FShowHint and Value then
    begin
      // Note: we assign self as owner for both components, so we don't need
      //       to destroy them explicitly.
      // create and setup the hint label
      FHintLbl := TLabel.Create(Self);
      with FHintLbl do
      begin
        Hide;
        AutoSize := false;
        BorderStyle := bsSingle;
        Layout := tlCenter;
        OnClick := DoOnClick;
      end;
      // create and setup the pause timer
      FTimer := TTimer.Create(Self);
      FTimer.Enabled := False;
      FTimer.OnTimer := DoOnTimer;
    end;
  FShowHint := Value;
end;

procedure THintListBox.DoOnTimer(Sender: TObject);
begin
  // the HintHidePause time has elapsed, we can hide the hint
  FHintLbl.Hide;
  FTimer.Enabled := False;
end;

end.
